/* ***************************************************** **
   ch10_creating_delimited_text.sql
   
   Skrypt dla książki Praktyczna nauka SQL dla Oracle, Helion (2022),
   napisanej przez Kima Berga Hansena, https://www.kibeha.dk
   Używasz na własną odpowiedzialność.
   *****************************************************
   
   Rozdział 10.
   Tworzenie ograniczonego tekstu
   
   Skrypt przeznaczony do wykonania w schemacie PRACTICAL
** ***************************************************** */

/* -----------------------------------------------------
   Konfiguracja formatowania sqlcl
   ----------------------------------------------------- */

set pagesize 80
set linesize 80
set sqlformat ansiconsole
set long 8000

/* -----------------------------------------------------
   Przykładowy kod do rozdziału 10.
   ----------------------------------------------------- */

-- Listing 10.2. Wyświetlenie listy browarów i kupionych w nich produktów

select *
from brewery_products
order by brewery_id, product_id;

-- Listing 10.3. Przykład użycia funkcji listagg() do utworzenia listy produktów

select
   max(brewery_name) as brewery_name
 , listagg(product_name, ',') within group (
      order by product_id
   ) as product_list
from brewery_products
group by brewery_id
order by brewery_id;

-- Zmiana kolejności listy

select
   max(brewery_name) as brewery_name
 , listagg(product_name, ',') within group (
      order by product_name
   ) as product_list
from brewery_products
group by brewery_id
order by brewery_id;

-- Listing 10.5. Przykład użycia wbudowanej funkcji collect() i utworzonej przez użytkownika

select
   max(brewery_name) as brewery_name
 , name_coll_type_to_varchar2(
      cast(
         collect(
            product_name
            order by product_id
         )
         as name_coll_type
      ) 
    , ','
   ) as product_list
from brewery_products
group by brewery_id
order by brewery_id;

-- Alternatywa używjąca apex_t_varchar2 i apex_string.join

select
   max(brewery_name) as brewery_name
 , apex_string.join(
      cast(
         collect(
            product_name
            order by product_id
         )
         as apex_t_varchar2
      ) 
    , ','
   ) as product_list
from brewery_products
group by brewery_id
order by brewery_id;

-- Listing 10.7. Przykład użycia niestandardowej funkcji agregacji stragg()

select
   max(brewery_name) as brewery_name
 , stragg(
      stragg_expr_type(product_name, ',')
   ) as product_list
from brewery_products
group by brewery_id
order by brewery_id;

-- Użycie niepotrzebnej klauzuli może spowodować uporządkowanie wyniku

select
   max(brewery_name) as brewery_name
 , stragg(
      distinct stragg_expr_type(product_name, ',')
   ) as product_list
from brewery_products
group by brewery_id
order by brewery_id;

-- Listing 10.8. Przykład użycia funkcji xmlagg() i wyodrębnienia tekstu z danych XML

select
   max(brewery_name) as brewery_name
 , rtrim(
      xmlagg(
         xmlelement(z, product_name, ',')
         order by product_id
      ).extract('//text()').getstringval()
    , ','
   ) as product_list
from brewery_products
group by brewery_id
order by brewery_id;

-- Pominięcie ekstrakcji przez uznanie treści jako xml

select
   max(brewery_name) as brewery_name
 , rtrim(
      xmlagg(
         xmlparse(content product_name || ',' wellformed)
         order by product_id
      ).getstringval()
    , ','
   ) as product_list
from brewery_products
group by brewery_id
order by brewery_id;

-- Listing 10.9. Komunikat o błędzie ORA-01489 wygenerowany w przypadku użycia funkcji listagg()

select
   listagg(rpad(p.name, 20)) within group (
      order by p.id
   ) as product_list
from products p
join monthly_sales ms
   on ms.product_id = p.id;

-- Listing 10.10. Wyeliminowanie błędu podczas używania funkcji listagg()

select
   listagg(
      rpad(p.name, 20)
      on overflow truncate '{more}' with count
   ) within group (
      order by p.id
   ) as product_list
from products p
join monthly_sales ms
   on ms.product_id = p.id;

-- Listing 10.11. Zmniejszenie za pomocą distinct powielających się danych

select
   listagg(distinct rpad(p.name, 20)) within group (
      order by p.id
   ) as product_list
from products p
join monthly_sales ms
   on ms.product_id = p.id;

-- Bonus: w wersji wcześniejszej niż 19c używamy funkcji stragg()

select
   stragg(
      distinct stragg_expr_type(rpad(p.name, 20), null)
   ) as product_list
from products p
join monthly_sales ms
   on ms.product_id = p.id;

-- Listing 10.12. Przykład użycia funkcji xmlagg() do agregacji wartości na postać typu clob

select
   xmlagg(
      xmlparse(
         content rpad(p.name, 20) wellformed
      )
      order by p.id
   ).getclobval() as product_list
from products p
join monthly_sales ms
   on ms.product_id = p.id;

-- Listing 10.13. Przykład użycia funkcji json_arrayagg() do agregacji danych na postać wartości typu clob

select
   json_value(
      replace(
         json_arrayagg(
            rpad(p.name, 20)
            order by p.id
            returning clob
         )
       , '","'
       , ''
      )
    , '$[0]' returning clob
   ) as product_list
from products p
join monthly_sales ms
   on ms.product_id = p.id;

-- Listing 10.14. Przykład użycia funkcji apex_string.join_clob() do agregacji danych na postać wartości typu clob

select
   apex_string.join_clob(
      cast(
         collect(
            rpad(p.name, 20)
            order by p.id
         )
         as apex_t_varchar2
      )
    , ''
    , 12 /* dbms_lob.call */
   ) as product_list
from products p
join monthly_sales ms
   on ms.product_id = p.id;

/* ***************************************************** */
